# Traversal & Query

## Introduction

Pawpaw's ``Ito`` class features recursive storage of descendants to arbitrary depth, resulting in a fully hierarchical data store.  The resulting tree
can be searched for relevant data using either manual traversal techniques, or a highly capable structured query syntax called *plumule*[^plumule].

## Manual Traversal

Manually traversing Pawpaw trees is possible using either the ``.children`` property to access an ``Ito`` object's children, or the ``.parent`` property to access its parent:

```python
>>> from pawpaw import Ito
>>> s = 'the quick brown fox'
>>> re = regex.compile(r'(?P<word>(?P<char>\w)+\W*)*')
>>> i = Ito.from_match(re.fullmatch(s))
>>> last_child = i.children[-1]  # traverse down to child
>>> str(last_child)  
'fox'
>>> str(last_child.parent)  # traverse up to parent
'the quick brown fox'
```

In addition to the ``.parent`` and ``.children``, three methods are available to facilitate traversal:

| Method                       | Description |
| :---                         | :--- |
| ``.get_root``                | Finds root of current node; returns None if the current node is itself a root |
| ``.walk_descendants``        | Performs a depth-first traversal of all descendants of current node |
| ``.walk_descendants_levels`` | Performs a depth-first traversal of all descendants of current node, yielding a ``Tuple[int, Ito]`` for each node whose integer is the depth from the starting node |

Example:

```python
>>> print('|'.join(str(d) for d in i.walk_descendants()))
the|t|h|e|quick|q|u|i|c|k|brown|b|r|o|w|n|fox|f|o|x
```

## Plumule Queries

Manually traversing Pawpaw trees is practical for small collections.  Larger collections,
by contrast, can benefit from Pawpaw's *plumule* query language.  Plumule has similar syntax and usage to 
[XPath](https://www.w3.org/TR/xpath/).  For example:

```python
>>> plumule_xpr = '*{*[s:o]}'  # words containing 'o'
>>> print([str(w) for w in i.find_all(plumule_xpr)])
['brown', 'fox']
```

Plumule queries are strings and offer several advantages:

* Easy string construction
* Persistence
* Can be run against multiple, *arbitrary* trees
 
Plumule queries are interpreted by Pawpaw's *radicle* query engine, which even allows pre-compilation to improve run-time
performance:

```python
>>> query = pawpaw.query.compile(plumule_xpr)
>>> print([str(w) for w in query.find_all(i)])
['brown', 'fox']
```

## Plumule Syntax

Plumule query sytax allows you to search for arbitrary nodes in an ``Ito`` Tree.  A Plumule query comprises a sequence of one or more *phrases* separated by fore-slash characters:

```
query := phrase [/ phrase] [/ phrase] ...
```

Phrases, in turn, consists of axes, filters, and subqueries:

* Phrases
  * Axes
  * Filters
  * Subqueries

Each component is further discussed below.

### Phrase

A phrase consists of an *axis* along with an optional *filters* and *subqueries* parts:

```
phrase := axis [filters] [subqueries]
```

### Axis

An axis consists of an optional *order operator*, a required *axis identifier*, and an optional *self option*:

```
axis := [order_operator] axis_identifier [self_option]
```

A Plumule axis defines a traversal step.  For example, the axis identifier ``*`` indicates that traversal
should proceed to the children of the current node(s).  For a given node, an axis may yield zero,
one, or more nodes, the default order of which will always be from nearest to farthest from the current node.

The order of nodes can be reversed by using a minus character (-) for the order operator.  A plus
character (+) indicates nearest-to-furthest ordering, and is the assumed default when the
order operator is not supplied.

#### Order Operator Values

| Value | Meaning                         |
|:-----:|:--------------------------------|
| ``-`` | Furthest-to-nearest             |
| ``+`` | Nearest-to-furthest *(default)* |

In some cases, it may be useful to return the current node as part traversal step.  For example,
the axis operator ``..`` indicates traversal to the parent node, however, if the current node is
a root, then there is no parent to traverse to.  The self option can be used to ensure that a
traversal step will always return something.  Two possible values are allows:

#### Self Option Values

| Value  | Returns Current Node If |
| :---:  | :--- |
| ``!``  | Traversal step is empty |
| ``!!`` | Always |

If a self option is supplied, the current node will always be returned first, unless
the axis is reversed with the '-' operator.

#### Axis Identifiers

|  Identifier  |     Meaning    | Description |
| :---         | :---           | :--- |
| ``'....'``     | root           | Furthest ancestor of the current node that has no ancestors itself  |
| ``'...'``      | ancestors      | All ancestors of the current node beginning with the parent and ending with the root |
| ``'..'``       | parent         | Nearest ancestor of the current node |
| ``'.'``        | self           | Reflection of the current node |
| ``'><'``       | de-dup         | Reflects current axis, but removes duplicate nodes |
| ``'*'``        | children       | Children of the current node |
| ``'**'``       | descendants    | All descendants of the current node, starting with the children and ending with leaves |
| ``'***'``      | leaves         | Descendants of the current node that have no descendants themselves |
| ``'<<<'``      | preceeding     | Nodes that are *before* the current node, *includes* prior siblings and *excludes* ancestors & descendants |
| ``'<<'``       | prior siblings | Nodes *before* the current node that have the same parent |
| ``'<'``        | prior sibling  | The nearest node *before* the current node that has the same parent |
| ``'>'``        | next sibling   |  The nearest node *after* the current node that has the same parent |
| ``'>>'``       | next siblings  |  Nodes *after* the current node that have the same parent |
| ``'>>>'``      | following      | Nodes that are *after* the current node, *includes* next siblings and *excludes* ancestors & descendants |

#### Examples

```python
 '*'    # all children; equivalent to iter(i.children)
 '*/*'  # all grandchildren; equivalent to iter(gc for c in i.children for gc in c.children)
 '../*' # all siblings and self: equivalent to i.parent.children
 '>>/*' # children of all successive siblings
```

### Filters

A filters expression consist of one or more filter components, separated with logical operators:

```
filters := [not_operator] filter [ combining_operator [not_operator] filter ]...
````

Parentheses can be used for grouping to override operator precedence as needed.

#### Operators

The *not operator* consists of the tilde character (``'~'``):

```
not_operator := '~'
```

A *combining operator* consists of either a ``'&'``, ``'^'``, or ``'|'`` character:

```
combining_operator := '&', '^', or '|'
```
 
The operators are listed below in order of precedence:

|  Operator  |  Combining  |  Meaning  |
| :---:      | :---:       | :---:     |
| ``'~'``    | ✗           | NOT       |
| ``'&'``    | ✓           | AND       |
| ``'^'``    | ✓           | XOR       |
| ``'\|'``   | ✓           | OR        |

#### Key-Value Pair

An individual filter consists key-value pair, surrounded by square brackets.  A key can be prefixed with the not operator ('~'), which has the same effect as if it occurs immediately before the opening bracket.

```
filter := '[' [NOT] key ':' value ']'
```

The filter key conists of a string that indicates what filtering action to perform on the current axis nodes.  The filter value
provides additional data to the filtering action.  Allowable values for keys and values are as follows:

| Key                   |          Alt Keys(s)          | Meaning                                                                 | Example(s)                             |
| :----:                |:-----------------------------:| :---                                                                    | :---                                   |
| ``'desc'``            |            ``'d'``            | One or more ``str`` values used to match againss ``.desc`` of axis; values must be separated with commas, literal commas must be escaped | ``[d:number]``<br />``[d:word,char]`` |
| ``'str'``          |            ``'s'``            | A ``str`` used to match against ``str()`` of axis | ``[s:foo]``<br />``[s:foo,goo]``|
| ``'str-casefold'`` |   ``'scf'``,<br />``'lcs'``   | Checks if casefolded ``str()`` of axis matches casefolded value | ``[scf:FoO,GoO]`` |
| ``'str-casefold-ew'`` | ``'scfew'``,<br />``'lcsew'`` | Checks if casefolded ``str()`` of axis ends with with casefolded value | ``[scfew:a,1]`` |
| ``'str-casefold-sw'`` | ``'scfsw'``,<br />``'lcssw'`` | Checks if casefolded ``str()`` of axis starts with with casefolded value | ``[scfsw:a,1]`` |
| ``'str-ew'`` |           ``'sew'``           | Checks if ``str()`` of axis ends with value | ``[scfew:a,1]`` |
| ``'str-sw'`` |           ``'ssw'``           | Checks if ``str()`` of axis starts with with value | ``[scfsw:a,1]`` |
| ``'index'`` |            ``'i'``            | One or more ``int`` tuples consisting of a *start* and optional *stop* values; matches against the enumeration index(ices) of the axis; start and stop must be non-negative integers; start and stop behave like a *slice*, i.e., stop should be *one more than* the last desired value; tuples must be separated with commas; *start* and *stop* must be separated with hyphens; a start followed by a hypen *without* a stop will be intepreted as ≥ , e.g. ``[i:5-]`` means *index(ices) greater than or equal to 5* | ``[i:1]``<br />``[i:2,3,4]``<br />``[i:2-3]``<br />``[i:2,5-7]``<br />``[i:0,5-,3]`` |
| ``'predicate'`` |            ``'p'``            | Key for filter function used to match against axis A ``str`` used as a key to entry in dictionary of type: ``typing.Dict[str, typing.Callable[pawpaw.Types.C_EITO, bool]]``  The value retrieved from the ``dict`` use used as a filter against the axis | ``[p:key1]``<br />``[p:key1,key2]`` |
| ``'value'`` |            ``'v'``            | A ``str`` used as a key to entry in dictionary of type::      typing.Dict[str, typing.Any]  The value retrieved from the ``dict`` is used to match against the ``.value()`` of the axis | ``[p:key]``<br />``[p:key1, key2]`` |

#### Parentheses

Plumule supports parentheses for to perform logical grouping:

```python
# all children having .desc equal to 'word' or 'number'
'*[d:word,number]'  

# all descendants with .desc not 'word', or with
# .desc 'word' and .__str__() 'ten'
'**[~d:word] | ([d:word] & [s:ten])'  
```

#### Predicates

Plumule supports arbitrary predicates as part of a query.  Predicates are refererred to in a plumule query via the ``'predicate'`` (or ``'p'``) key, whose associated value(s) are keys in a dictionary of type:

```python
typing.Dict[str, typing.Callable[pawpaw.Types.C_EITO, bool]] 
```

Predicates allow you to use your own code as part of a query filter:

```python
>>> import pawpaw
>>> ito = pawpaw.Ito('ABcd12Ef')
>>> ito.children.add(*ito)
>>> query = '*[p:isnumeric]'
>>> predicates = {}
>>> predicates['isnumeric'] = lambda ei: ei.ito.str_isnumeric()
>>> for i in ito.find_all(query, predicates=predicates):
...     print(i)
1
2
```

### Subqueries

The subqueries component of a phrase consists of one or more subquery expressions, separated with the same logical operators used for filters, and grouped as needed with parentheses:

```
subqueries := [not_operator] subquery [ combining_operator [not_operator] subquery ]...
```

A subquery is itself a plumule query, contained in curly braces:

```
subquery := '{' query '}'
```

Whereas the filters component constrains nodes of the current axis that meet a given criteria, the subqueries component constrains the nodes of the current axis based on a *relative query*.  Without the use of a relative query, for a given axis you would first have to traverse away from it, perform a filter, and then *traverse backwards* back to the start axis.  You can actually do this with plumule in certain scenarios.  For example, if you wanted to find all grandchildren having descriptor values of ``'foo'``, you could use this filter:

```python
'*/*[d:foo]/../../><'  # children → children with .desc == 'foo' → parent → parent → dedup 
```

Note that you need to use the de-dup operator, ``'><'`` to deal with nodes having *more than one* matching grandchild.  In addition, this approach is limited for situations where you traverse known path distance, and when the traversal and return don't change ordering.   As an example, consider if you wanted _all_ descendants  with descriptor values of ``'foo'``, instead of just grandchildren:

```python
'**[d:foo]/???'  # How to get back?
```

Because you might be any number of steps down the hierarchy when you find a matching node, you don't know the number of steps to get back to where you started.  You *might* be able to construct another filter and apply it to a subsequent ancestor axis:

```python
'**[s:abc]/...[d:something]/><'
```

However, this only works if there exists a unique criteria for your starting point, which might be in the middle of a larger query.  What you need is a way to say "select the nodes I currently have based on their ancestors or descendants", without having to worry about traversing away and back.  And this is exactly what a subquery allows you to do:

```python
'.{*/*[d:foo]}'  # Reflect nodes in current axis whose grandchildren have .desc == 'foo'
'.{**[d:foo]}'   # Reflect nodes in current axis having descendants with .__str__() == 'abc'
```

The behavior of a subquery is analogous to bookmarking the current axis, performing an exploratory query for each node in it, and then filtering the axis to select only those nodes whose exploratory query yielded any results.  You can even use a subquery to filter based on *non-matches*, e.g.:

```python
'.~{**[d:foo]}'  # Reflect nodes in current axis that DON'T have ancestors with .desc == 'foo'
```

Do take note that the following query is not the same:

```python
'.{**~[d:foo]}'  # This is NOT the same as '.~{**[d:foo]}'
```

In the first case, the not operator is applied to the results of the subquery, inverting it.  Logically, you can think of these two queries as:

```python
'.~{**[d:foo]}'  # NOT (has descendant with .desc == 'foo')
'.{**~[d:foo]}'  # (has descendant with .desc != 'foo')
```

which are two very different things indeed!

#### Examples

```python
'*{*}'  # children that themselves have at least one child
'.{**[d:word]}'  # nodes having descendants with .desc == 'word'
'.[d:digit]{**{[s:4,6] | ~[d:prime]} | !{..[d:sci]}'  # nodes with .desc == 'digit' and having a) descendants with substr in ['4', '6'] or b) ancestors with .desc = 'sci'
```

## Tips & Tricks

Q: How can I 'OR' together filter and subquery components?

A: Operators are not supported between the filters and subqueries components of a query phrase.  However, you can achieve the same result by moving the filter to a subquery::

```
[my_filter]{my_subquery}  ->  {./[my_filter]} | {my_subquery}
```

[^plumule]: **plumule**: *the rudimentary shoot or stem of an embryo plant.*  This name was chosen because Pawpaw query syntax is based on axial path statements - each one of which is akin to travelling up or down a dendritic root structure.
